<?php

namespace ShopWP_Webhooks;

use ShopWP\Utils;
use ShopWP\Factories;

class Plugin {

	public static $_instance = null;
	public $Webhooks = null;
	public $settings;
	public $Processing;
	public $GraphQL;
	public $API_Products;
	public $API_Items_Products;
	public $API_Admin_Webhooks;
	public $DB_Settings_Connection;
	public $CPT_Model;

	public static function instance() {
		
		if ( is_null( self::$_instance ) ) {
			self::$_instance = new self();
		}

		return self::$_instance;
	}

	public function __construct() {

		$this->settings 				= Factories\DB\Settings_Plugin_Factory::build();
		$this->Webhooks 				= Factories\Webhooks_Factory::build($this->settings);
		$this->Processing 				= Factories\Processing\Items_Factory::build();
		$this->GraphQL					= Factories\API\GraphQL_Factory::build($this->settings);
		$this->API_Items_Products 		= Factories\API\Items\Products_Factory::build($this->settings);
		$this->API_Admin_Webhooks 		= Factories\API\Admin\Webhooks\Webhooks_Factory::build($this->settings);
		$this->DB_Settings_Connection 	= Factories\DB\Settings_Connection_Factory::build();
		$this->CPT_Model 				= Factories\CPT_Model_Factory::build();

		add_action('rest_api_init', [$this, 'register_routes']);
		
		/*
		
		These webhooks will only fire when modifications are made to products assigned to the ShopWP sales channel
		
		*/
		add_action('shopwp_webhook_after_product_delete', [$this, 'on_product_delete']);
		add_action('shopwp_webhook_after_product_create', [$this, 'on_product_create']);
		add_action('shopwp_webhook_after_product_update', [$this, 'on_product_update']);
		
		add_action('shopwp_webhook_after_collection_delete', [$this, 'on_collection_delete']);
		add_action('shopwp_webhook_after_collection_create', [$this, 'on_collection_create']);
		
	}

	/*
	
	If this runs, we can assume the following:
	
	1. The product is assigned to the ShopWP sales channel
	2. The product is not deleted in Shopify 
	3. The status is 'active' because it will not run when the Shopify product status changes

	Sometimes ShopWP will be configured to only sync products by a certain criteria. For example, sync products by X tag or from Y collection.

	If a product changes inside Shopify that match these requirements (for example the required tag is removed), then this function needs to make sure the product is deleted (or created).

	To do this, we can take our syncing query and run it against the Shopify Admin API. If it returns a payload, we know the product still passes the syncing query test. If it
	does not return anything, we know the product does not pass the test and therefore, can be deleted from WordPress.

	We're setting a "reasonable" amount of time (10 seconds) to wait before making our API request to check whether the product still exists.

	*/
	public function on_product_update($data) {

		if (empty($data) || empty($data->product_listing)) {

			Utils::log([
				'thing_to_log'  => __('No data found on product update webhook!', 'shopwp'),
				'log_level'     => 'warning',
				'file_name'     => 'shopwp-debug.log'
			]);
			return;
		}

		if ($this->Processing->DB_Settings_Syncing->is_syncing()) {

			$msg = 'Not processing Product Update webhook for product: ' . $data->product_listing->title . ' because an existing sync is occurring.';

			Utils::log([
				'thing_to_log'  => $msg,
				'log_level'     => 'warning',
				'file_name'     => 'shopwp-debug.log'
			]);
			return;
		}

		$bulk_query 			= $this->get_bulk_query($data);
		$selective_sync 		= $this->Processing->DB_Settings_General->selective_sync_status();
		$sync_by_collections    = $this->Processing->DB_Settings_General->sync_by_collections();

		$products_query 		= $this->API_Items_Products->get_products_query($bulk_query, $selective_sync, $sync_by_collections, true);
		
		$final_query = 'query {' . 
			$products_query 
		. '}';

		set_time_limit(0);

		// The product is not updated in time before this webhook runs. We'll wait a reasonable amount of time.
		sleep(15);

		// Fetch the product using the bulk query. If it returns something, create / update the product. If not, delete.
		$resp = $this->GraphQL->graphql_api_request(
			[
				"query" => $final_query
			],
			'admin'
		);

		if (is_wp_error($resp)) {

			Utils::log([
				'thing_to_log'  => $resp->get_error_message(),
				'log_level'     => 'error',
				'file_name'     => 'shopwp-debug.log'
			]);

			return;
		}

		$product_id 		= $data->product_listing->product_id;
		$product_title 		= $data->product_listing->title;

		// Product should be deleted
		if (empty($resp->products->edges)) {

			$post_id = $this->CPT_Model->get_post_id_by_item_id($product_id);

			if (empty($post_id)) {

				Utils::log([
					'thing_to_log'  => __('Product "' . $product_title . '" should be deleted during PRODUCT_LISTINGS_UPDATE webhook event, but is already deleted in WordPress. Doing nothing.', 'shopwp'),
					'log_level'     => 'warning',
					'file_name'     => 'shopwp-debug.log'
				]);

				return;

			} else {


				Utils::log([
					'thing_to_log'  => __('Deleting product "' . $product_title . '" during PRODUCT_LISTINGS_UPDATE webhook event.', 'shopwp'),
					'log_level'     => 'warning',
					'file_name'     => 'shopwp.log'
				]);

				$result = $this->delete_product($product_id, $post_id);

			}

		// Product should be updated
		} else {

			/*

			Important so we don't log any irrelevant syncing errors.

			This turns on is_syncing, so we need to call expire_sync if we bail before the process begins.4

			*/
			$reset = $this->Processing->DB_Settings_Syncing->get_system_ready_for_sync();
			
			Utils::log([
				'thing_to_log'  => __('Creating or updating product "' . $product_title . '" during PRODUCT_LISTINGS_UPDATE webhook event.', 'shopwp'),
				'log_level'     => 'info',
				'file_name'     => 'shopwp-debug.log'
			]);
			
			$product = $resp->products->edges[0]->node;

			if (!empty($sync_by_collections)) {

				if (empty($product->collections) || empty($product->collections->edges)) {


					Utils::log([
						'thing_to_log'  => __('Syncing by collections, but product "' . $product_title . '" not assigned to any. Skipping sync.', 'shopwp'),
						'log_level'     => 'info',
						'file_name'     => 'shopwp-debug.log'
					]);

					$this->Processing->DB_Settings_Syncing->expire_sync();
					
					return;
				}
				
				$found = $this->CPT_Model->does_product_exist_in_collection($sync_by_collections, $product->collections->edges, false);

				if (empty($found)) {

					Utils::log([
						'thing_to_log'  => __('Product "' . $product_title . '" is assigned collections, but does not belong to required ones. Skipping update.', 'shopwp'),
						'log_level'     => 'info',
						'file_name'     => 'shopwp-debug.log'
					]);

					$this->Processing->DB_Settings_Syncing->expire_sync();

					return;
				}
			}

			$this->Processing->process($product, true);

		}

	}

	public function get_bulk_query($data) {

		/*

		If a product already exists in the database, it will be updated during the processing step instead of duplicated

		*/
		$query_from_db 			= $this->Processing->DB_Settings_General->col('bulk_products_query', 'string');
		$sync_by_collections 	= $this->Processing->DB_Settings_General->sync_by_collections();

		// If the user has set a syncing query, use that with ID. Otherwise fetch by ID only
		if ($query_from_db === '*' || !empty($sync_by_collections)) {
			$bulk_query = 'id:' . $data->product_listing->product_id;

		} else {

			// This ensures that only the specific product is synced, IF it also matches the given syncing query 
			$bulk_query = $query_from_db . ' AND id:' . $data->product_listing->product_id;
		}

		return $bulk_query;

	}

	/*
	
	Will run if newly created, added to the ShopWP sales channel, or status is changed to active
	
	*/
	public function on_product_create($data) {

		if (empty($data) || empty($data->product_listing)) {

			Utils::log([
				'thing_to_log'  => __('No data found on product create webhook!', 'shopwp'),
				'log_level'     => 'warning',
				'file_name'     => 'shopwp-debug.log'
			]);

			return;
		}

		if ($this->Processing->DB_Settings_Syncing->is_syncing()) {


			$msg = 'Not processing Product Create webhook for product: ' . $data->product_listing->title . ' because an existing sync is occurring.';

			Utils::log([
				'thing_to_log'  => __($msg, 'shopwp'),
				'log_level'     => 'warning',
				'file_name'     => 'shopwp-debug.log'
			]);
			return;
		}

		$bulk_query 			= $this->get_bulk_query($data);

		$selective_sync 		= $this->Processing->DB_Settings_General->selective_sync_status();
		$sync_by_collections    = $this->Processing->DB_Settings_General->sync_by_collections();

		$products_query 		= $this->API_Items_Products->get_products_query($bulk_query, $selective_sync, $sync_by_collections, true);

		$final_query = 'query {' . 
			$products_query 
		. '}';

		set_time_limit(0);

		// The product is not updated in time before this webhook runs. We'll wait a reasonable amount of time.
		sleep(15);

		// Fetch the product using the bulk query. If it returns something, create / update the product. If not, delete.
		$resp = $this->GraphQL->graphql_api_request(
			[
				"query" => $final_query
			],
			'admin'
		);

		if (is_wp_error($resp)) {


			Utils::log([
				'thing_to_log'  => $resp->get_error_message(),
				'log_level'     => 'error',
				'file_name'     => 'shopwp-debug.log'
			]);

			return;
		}

		/*

		Important so we don't log any irrelevant syncing errors.

		*/
		$reset = $this->Processing->DB_Settings_Syncing->get_system_ready_for_sync();

		Utils::log([
			'thing_to_log'  => __('Creating product "' . $data->product_listing->title . '" during PRODUCT_LISTINGS_ADD webhook event.', 'shopwp'),
			'log_level'     => 'info',
			'file_name'     => 'shopwp.log'
		]);

		if (empty($resp->products->edges)) {

			$this->Processing->DB_Settings_Syncing->expire_sync();
			return;
		}


		$product_data = $resp->products->edges[0]->node;

		$this->Processing->DB_Settings_Syncing->update_col('syncing_totals_products', 1);

		// If user is not syncing products by collection, then just process the product instead
		if (empty($sync_by_collections)) {

			$this->Processing->process($product_data, true);

		} else {

			if (empty($product_data->collections) || empty($product_data->collections->edges)) {

				Utils::log([
					'thing_to_log'  => __('Syncing by collections, but product "' . $data->product_listing->title . '" not assigned to any. Skipping sync.', 'shopwp'),
					'log_level'     => 'info',
					'file_name'     => 'shopwp-debug.log'
				]);

				$this->Processing->DB_Settings_Syncing->expire_sync();
				
				return;
			}

			$collection_ids 	= $this->CPT_Model->create_list_of_collection_ids($product_data);
			$found 				= $this->CPT_Model->does_product_exist_in_collection($sync_by_collections, $collection_ids, false);

			if (empty($found)) {

				Utils::log([
					'thing_to_log'  => __('Product "' . $data->product_listing->title . '" is assigned collections, but does not belong to required ones. Skipping update.', 'shopwp'),
					'log_level'     => 'info',
					'file_name'     => 'shopwp-debug.log'
				]);

				$this->Processing->DB_Settings_Syncing->expire_sync();

				return;
			}

			$this->Processing->process($product_data, true);

		}

	}

	public function delete_product($product_id, $post_id) {
		return [
			'products' 		=> $this->Processing->DB_Products->delete_products_from_product_id($product_id),
			'images' 		=> $this->Processing->DB_Images->delete_images_from_product_id($product_id, $post_id),
			'tags' 			=> $this->Processing->DB_Tags->delete_tags_from_product_id($product_id),
			'variants' 		=> $this->Processing->DB_Variants->delete_variants_from_product_id($product_id),
			'metafields' 	=> $this->Processing->DB_Metafields->delete_metafields_from_product_id($product_id),
			'options' 		=> $this->Processing->DB_Options->delete_options_from_product_id($product_id),
			'post' 			=> \wp_delete_post($post_id, true)
		];		
	}

	/*
	
	Removes the product (and all it's data) from WP when deleted in Shopify

	Will run if fully deleted, removed from ShopWP sales channel, or status is changed to draft

	*/
	public function on_product_delete($data) {
		
		if (empty($data) || empty($data->product_listing)) {

			Utils::log([
				'thing_to_log'  => __('No data found on product delete webhook!', 'shopwp'),
				'log_level'     => 'warning',
				'file_name'     => 'shopwp-debug.log'
			]);

			return;
		}

		if ($this->Processing->DB_Settings_Syncing->is_syncing()) {

			$msg = 'Not processing Product Delete webhook for product: ' . $data->product_listing->title . ' because an existing sync is occurring.';

			Utils::log([
				'thing_to_log'  => $msg,
				'log_level'     => 'warning',
				'file_name'     => 'shopwp-debug.log'
			]);
			return;
		}


		$product_id 			= $data->product_listing->product_id;
		$post_id 				= $this->CPT_Model->get_post_id_by_item_id($product_id);

		if (empty($post_id)) {

			Utils::log([
				'thing_to_log'  => __('No product found in database on product delete webhook, nothing to do.', 'shopwp'),
				'log_level'     => 'warning',
				'file_name'     => 'shopwp-debug.log'
			]);

			return;
		}

		$resp = $this->API_Items_Products->fetch_product_from_rest($product_id);

		if (is_wp_error($resp)) {

			Utils::log([
				'thing_to_log'  => __('Product delete webhook error: ', 'shopwp') . $resp->get_error_message(),
				'log_level'     => 'error',
				'file_name'		=> 'shopwp-debug.log'
			]);

			return;

		}

		$this->delete_product($product_id, $post_id);

		Utils::log([
			'thing_to_log'  => __('Shopify product with id: ' . $product_id . ' was deleted from WordPress.', 'shopwp'),
			'log_level'     => 'info',
			'file_name'		=> 'shopwp-debug.log'
		]);

	}


	
	public function on_collection_create($data) {

		if (empty($data) || empty($data->collection_listing)) {

			Utils::log([
				'thing_to_log'  => __('No data found on collection create webhook!', 'shopwp'),
				'log_level'     => 'warning',
				'file_name'     => 'shopwp-debug.log'
			]);

			return;
		}

		if ($this->Processing->DB_Settings_Syncing->is_syncing()) {

			Utils::log([
				'thing_to_log'  => __('Not processing Collection Create webhook for collection: ', 'shopwp') . $data->collection_listing->title . '. ' . __('An existing sync is already occurring.', 'shopwp'),
				'log_level'     => 'warning',
				'file_name'     => 'shopwp-debug.log'
			]);
			return;
		}

		$collection_id = 'gid://shopify/Collection/' . $data->collection_listing->collection_id;

		$final_query = 'query {
			collection(id: "' . $collection_id . '") {
				id
				title
				handle
				updatedAt
				publishedOnCurrentPublication
				description
				descriptionHtml
				productsCount {
					count
				}
				ruleSet {
					appliedDisjunctively
					rules {
						condition
						column
						relation
						conditionObject
					}
				}
				seo {
					title
					description
				}
				legacyResourceId
				image {
					altText
					height
					width
					url
					src
					originalSrc
				}
			}
		}';

		set_time_limit(0);

		// The product is not updated in time before this webhook runs. We'll wait a reasonable amount of time.
		sleep(3);

		// Fetch the product using the bulk query. If it returns something, create / update the product. If not, delete.
		$resp = $this->GraphQL->graphql_api_request(
			[
				"query" => $final_query
			],
			'admin'
		);

		if (is_wp_error($resp)) {

			Utils::log([
				'thing_to_log'  => __('Collection create webhook error: ', 'shopwp') . $resp->get_error_message(),
				'log_level'     => 'error',
				'file_name'		=> 'shopwp-debug.log'
			]);

			return;

		}

		if (!empty($resp->collection)) {

			/*

			Important so we don't log any irrelevant syncing errors.

			*/
			$reset = $this->Processing->DB_Settings_Syncing->get_system_ready_for_sync();

			Utils::log([
				'thing_to_log'  => __('Creating collection "' . $data->product_listing->title . '" during COLLECTION_LISTINGS_ADD webhook event.', 'shopwp'),
				'log_level'     => 'info',
				'file_name'     => 'shopwp.log'
			]);

			
			$this->Processing->process($resp->collection, true);
		}

	}

	/*
	
	Removes the collection (and all it's data) from WP when deleted in Shopify

	*/
	public function on_collection_delete($data) {

		if (!empty($data)) {

			$collection_id = $data->collection_listing->collection_id;
			
			$delete_collection_custom_tables = $this->Processing->DB_Collections->delete_collections_from_collection_id($collection_id);

			$args = [
				'post_type' 	=> 'wps_collections',
				'meta_key' 		=> 'collection_id',
				'meta_value' 	=> $collection_id,
				'meta_compare' 	=> '='
			];

			$collection_posts = get_posts($args);

			$results = [];

			foreach ($collection_posts as $post) {
				$results[] = \wp_delete_post($post->ID, true);
			}

		}

	}

	/*
	
	The webhooks should be in this form: /wp-json/shopwp/v1/webhooks?topic=product_listings_add
	
	*/
	// public function add_webhook_routes($topics) {
		
	// }

	public function register_routes() {
		$this->Webhooks->API_Wrapper->api_route('/webhooks/delete', 'POST', [$this, 'handle_disconnect_webhooks']);
		$this->Webhooks->API_Wrapper->api_route('/webhooks/connect', 'POST', [$this, 'handle_connect_webhooks']);
		$this->Webhooks->API_Wrapper->api_route('/webhooks', 'POST', [$this, 'on_webhook_fire']);
	}

	public function handle_connect_webhooks($request) {
		
		$topics = $request->get_param('topics');
		$topics	= empty($topics) ? null : $topics;

		$topics = $this->connect_webhooks($topics);

		\wp_send_json_success($topics);
	}

	public function return_notice($results, $type) {

		$errors = [];

		foreach ($results as $key => $result) {
			if (is_wp_error($result)) {
				$errors[] = __('Failed to connect ' . $key . '. ' . $result->get_error_message(), 'shopwp');
			}
		}

		if (empty($errors)) {
			return \wp_send_json_success('Successfully ' . $type . ' ' . count(array_filter($results)) . ' webhooks!');
		}

		return \wp_send_json_success(implode("<br>", $errors));
	}

	public function disconnect_webhooks($params) {

		$callbackUrl 		= empty($params['callbackUrl']) ? null : $params['callbackUrl'];
		$topics 			= empty($params['topics']) ? null : $params['topics'];

		// While the webhook subscription may be empty, there could still be one connected. If this happens, we need cancel it and reconnect
		$all_webhook_subscriptions = $this->API_Admin_Webhooks->admin_api_get_all_webhook_subscriptions(['callbackUrl' => $callbackUrl, 'topics' => $topics]);

		if (is_wp_error($all_webhook_subscriptions)) {
			return $all_webhook_subscriptions;
		}

		if (empty($all_webhook_subscriptions->edges)) {
			return false;
		}

		$results = [];

		foreach ($all_webhook_subscriptions->edges as $key => $webhook) {

			if (is_wp_error($webhook)) {
				$results[] = $webhook;
				continue;
			}

			$resp = $this->admin_api_delete_webhook($webhook->node->id);

			if (is_wp_error($resp)) {
				$results[$key] = $resp;
				
			} else if (!empty($resp) && isset($resp->webhookSubscriptionDelete)) {
				$results[$key] = $resp->webhookSubscriptionDelete->deletedWebhookSubscriptionId;
			}
		}

		return $results;

	}

	public function handle_disconnect_webhooks($request) {

		$topics 			= $request->get_param('topics');
		$topics				= empty($topics) ? null : $topics;
		$routes 			= SHOPWP_AVAILABLE_WEBHOOKS;
		$results 			= [];
		$callbacks 			= [];

		// Disconnect all webhooks
		if (empty($topics)) {

			foreach ($routes as $route) {
				$callbacks[$route['value']] = $this->API_Items_Products->create_callback_url(['is_bulk' => false, 'old_topic' => $route['value']]);
			}

			$callbacks['webhooks'] = $this->API_Items_Products->create_callback_url(['is_bulk' => false]);

			foreach ($callbacks as $endpoint) {

				$result = $this->disconnect_webhooks([
					'callbackUrl' => $endpoint,
				]);

				if (is_array($result)) {
					foreach ($result as $webhook_removed) {
						$results[] = $webhook_removed;
					}
				} else {
					$results[$endpoint] = $result;
				}
			}

			$this->Webhooks->DB_Settings_General->update_col('connected_webhooks', '');

		// Disconnect one webhook
		} else {

			$results[$topics[0]] = $this->disconnect_webhooks([
				'callbackUrl' 	=> $this->API_Items_Products->create_callback_url(['is_bulk' => false]),
				'topics' 		=> $topics
			]);

			$all_webhooks 				= $this->Webhooks->DB_Settings_General->col('connected_webhooks', 'string');
			$connected_webhooks_in_db 	= \maybe_unserialize($all_webhooks);

			if (is_array($connected_webhooks_in_db)) {
				$updated = array_filter($connected_webhooks_in_db, function($val, $key) use($topics) {

					return $val['topic'] !== $topics[0];
					
				}, ARRAY_FILTER_USE_BOTH);

				if (empty($updated)) {
					$this->Webhooks->DB_Settings_General->update_col('connected_webhooks', '');
				} else {
					$this->Webhooks->DB_Settings_General->update_col('connected_webhooks', \maybe_serialize($updated));
				}
			}

		}

		return $this->return_notice($results, 'removed');
	}

	public function admin_api_get_webhook_by_callback_and_topic($topics, $callbackUrl) {
        return $this->GraphQL->graphql_api_request(
            $this->graph_query_get_webhooks_by_callback_and_topic($topics, $callbackUrl),
            'admin'
        );
    }

	public function admin_api_get_webhooks() {

        return $this->GraphQL->graphql_api_request(
            $this->graph_query_list_all_active_webhooks(),
            'admin'
        );

    }

	public function graph_query_list_all_active_webhooks() {
		return [
			"query" => 'query {
					webhookSubscriptions(first:100) {
						nodes {
							createdAt
							endpoint
							format
							id
							includeFields
							legacyResourceId
							metafieldNamespaces
							privateMetafieldNamespaces
							topic
							updatedAt								
						}
					}
				}
			',
		];
    }

	public function graph_query_get_webhooks_by_callback_and_topic($topics, $callbackUrl) {
		return [
			"query" => 'query {
					webhookSubscriptions(first:100, topics: ' . $topics . ', callbackUrl: "' . $callbackUrl . '") {
						nodes {
							createdAt
							endpoint
							format
							id
							includeFields
							legacyResourceId
							metafieldNamespaces
							privateMetafieldNamespaces
							topic
							updatedAt								
						}
					}
				}
			',
		];
    }

	public function graph_query_delete_webhook($webhook_id) {
		return [
			"query" => 'mutation webhookSubscriptionDelete($id: ID!) {
				webhookSubscriptionDelete(id: $id) {
					deletedWebhookSubscriptionId
					userErrors {
						field
						message
					}
				}
			}',
			"variables" => [
				'id' => $webhook_id
			]
		];
		
    }

	public function graph_query_register_webhook($webhook_body) {
		return [
			"query" => 'mutation webhookSubscriptionCreate($topic: WebhookSubscriptionTopic!, $webhookSubscription: WebhookSubscriptionInput!) {
				webhookSubscriptionCreate(topic: $topic, webhookSubscription: $webhookSubscription) {
					userErrors {
						field
						message
					}
					webhookSubscription {
						id
						endpoint
						topic
					}
				}
			}',
			"variables" => [
				'topic' => $webhook_body['webhook']['topic'],
				'webhookSubscription' => [
					'callbackUrl' => $webhook_body['webhook']['address'],
					'format' => $webhook_body['webhook']['format']
				]
			]
		];
		
    }

	public function admin_api_register_webhook($webhook_body) {

		$query = $this->graph_query_register_webhook($webhook_body);

        return $this->GraphQL->graphql_api_request(
            $query,
            'admin'
        );
    }

	public function admin_api_delete_webhook($webhook_id) {

        return $this->GraphQL->graphql_api_request(
            $this->graph_query_delete_webhook($webhook_id),
            'admin'
        );

    }

	public function find_label_from_topic($topic) {

		$all_webhooks = SHOPWP_AVAILABLE_WEBHOOKS;

		$found_webhook = array_values(array_filter($all_webhooks, function($webhook) use($topic) {
			return $webhook['value'] === strtoupper($topic);
		}));

		if (!empty($found_webhook)) {
			return $found_webhook[0]['label'];
		}

		return Utils::wp_error('No label found for topic: ' . $topic);

	}

	public function maybe_update_connected_webhooks_locally($webhooks_to_update) {

		$existing_connected_webhooks = $this->Webhooks->DB_Settings_General->col('connected_webhooks', 'string');
		$existing_connected_webhooks = \maybe_unserialize($existing_connected_webhooks);

		if (empty($existing_connected_webhooks)) {
			$existing_connected_webhooks = [];
		}

		$final_results = array_values(array_merge($existing_connected_webhooks, $webhooks_to_update));

		return $this->Webhooks->DB_Settings_General->update_col('connected_webhooks', \maybe_serialize($final_results));
	}

	/*
	
	These endpoints are set dynamically depending on what the user chooses 
	in the Webhooks settings.

	The webhook "topic" below is used for the end of the url. A full list of topics can 
	be found here: https://shopify.dev/docs/api/admin-graphql/2023-10/enums/WebhookSubscriptionTopic

	New webhooks URL (Topic is sent in a POST Request header from Shopify)
	https://82503ce103f7.ngrok.app/wp-json/shopwp/v1/webhooks

	Old webhooks URL: 
	https://82503ce103f7.ngrok.app?shopify_webhook=<topic_in_lowercase>

	*/
	public function connect_webhooks($topics)
    {
		if ($topics) {
			$topics_to_connect 		= $topics;
		} else {
			$topics_to_connect 		= $this->Webhooks->plugin_settings['general']['sync_by_webhooks'];
		}
        
		$webhooks_url 				= $this->Webhooks->plugin_settings['general']['url_webhooks'];
        $topics_unserialized 		= maybe_unserialize($topics_to_connect);

        if (empty($topics_unserialized)) {
			return \wp_send_json_error(__('No webhooks are selected to connect. Open the ShopWP syncing settings and select some.', 'shopwp'));
        }

		if (empty($webhooks_url)) {
			return \wp_send_json_error(__('The webhooks url is empty. Open the ShopWP syncing settings and add your WordPress domain.', 'shopwp'));
        }

		$results 					= [];
		$all_available_webhooks 	= SHOPWP_AVAILABLE_WEBHOOKS;

		foreach ($topics_unserialized as $topic) {

			$topic_lower = strtolower($topic);

			$full_callback = $this->API_Items_Products->create_callback_url(['is_bulk' => false]);

			$resp = $this->admin_api_register_webhook([
				"webhook" => [
					"address" 	=> $full_callback,
					"topic" 	=> $topic,
					"format" 	=> "JSON"
				]
			]);

			if (is_wp_error($resp)) {
				$results[] = false;

				Utils::log([
					'thing_to_log'  => $resp->get_error_message(),
					'log_level'     => 'error',
					'file_name'     => 'shopwp-debug.log'
				]);

			} else {

				if (!empty($resp->webhookSubscriptionCreate->userErrors)) {
					
					// Need to remove this topic and try adding it again
					if (str_contains($resp->webhookSubscriptionCreate->userErrors[0]->message, 'Address for this topic')) { 

						Utils::log([
							'thing_to_log'  => $resp->webhookSubscriptionCreate->userErrors[0]->message,
							'log_level'     => 'error',
							'file_name'     => 'shopwp-debug.log'
						]);

						$wh = $this->admin_api_get_webhook_by_callback_and_topic($topic, $full_callback);

						if (is_wp_error($wh)) {
							
							Utils::log([
								'thing_to_log'  => $wh->get_error_message(),
								'log_level'     => 'error',
								'file_name'     => 'shopwp-debug.log'
							]);

						} else if (empty($wh) || empty($wh->webhookSubscriptions) || empty($wh->webhookSubscriptions->nodes)) {

							Utils::log([
								'thing_to_log'  => 'ShopWP Error: No webhook found for topic: ' . $topic . ' at url: ' . $full_callback,
								'log_level'     => 'error',
								'file_name'     => 'shopwp-debug.log'
							]);

						} else {

							$to_add = [
								'id' 	=> $wh->nodes[0]->id,
								'label' => $this->find_label_from_topic($topic),
								'topic' => $topic
							];

							$updated_connected_webhooks_result = $this->maybe_update_connected_webhooks_locally($to_add);

							if ($updated_connected_webhooks_result) {
								
								Utils::log([
									'thing_to_log'  => __('Successfully added the existing webhook into the connected_webhooks column', 'shopwp'),
									'file_name'     => 'shopwp-debug.log'
								]);

								$results[] = $to_add;

							} else {
								Utils::log([
									'thing_to_log'  => __('Unable to update the connected_webhooks column. You may need to manually delete and reconnect.', 'shopwp'),
									'log_level'     => 'warning',
									'file_name'     => 'shopwp-debug.log'
								]);
							}

						}

					}

				} else {

					$label = $this->find_label_from_topic($topic);

					$webhook_id = $resp->webhookSubscriptionCreate->webhookSubscription->id;

					$results[] = [
						'id' 	=> $webhook_id,
						'label' => $label,
						'topic' => $topic
					];

				}
				
			}
			
		}

		if (empty($results)) {

			Utils::log([
				'thing_to_log'  => __('No webhooks were connected', 'shopwp'),
				'log_level'     => 'warning',
				'file_name'     => 'shopwp-debug.log'
			]);

		} else {
			$this->maybe_update_connected_webhooks_locally($results);
		}

		return $this->return_notice($results, 'connected');

    }

	public function on_webhook_fire($request) {

		$topic = $request->get_header('X-Shopify-Topic');

		$topic = Utils::convert_slash_to_underscore($topic);

		$this->webhook_process($topic);

	}

	public function webhook_process($webhook_name) {

		$decoded_data = $this->on_webhook_callback();

		if (!$decoded_data) {
			return;
		}

		$this->Webhooks->Template_Loader
                ->set_template_data($decoded_data, 'data', false)
                ->get_template_part('webhooks/' . $webhook_name);
	}

	public function on_webhook_callback() {

		$json_data                  = $this->Webhooks->before_webhook_process();
        $allow_insecure_default     = $this->Webhooks->DB_Settings_General->col('allow_insecure_webhooks', 'bool');
        $allow_insecure_webhooks    = apply_filters('shopwp_skip_bulk_webhook_ver', $allow_insecure_default);

        if (!$allow_insecure_webhooks && !$this->Webhooks->is_valid_webhook($json_data)) {

            $this->Webhooks->DB_Settings_Syncing->log(__('The webhook from Shopify is either invalid or expired.', 'shopwp'), 'error');
            $this->Webhooks->DB_Settings_Syncing->expire_sync();
            return false;
        }

		return json_decode($json_data);

	}

}

Plugin::instance();