<?php
/**
 * Plugin Name: ShopWP Webhooks
 * Description: ShopWP extension for managing Shopify webhooks
 * Plugin URI:  https://wpshop.io/extensions/webhooks
 * Version:     1.3.6
 * Author:      ShopWP
 * Author URI:  https://wpshop.io/extensions/webhooks
 * Text Domain: shopwp-webhooks
 */

if ( ! defined( 'ABSPATH' ) ) exit; // Exit if accessed directly

if (!defined('SHOPWP_WEBHOOKS_MODULE_DIR')) {
   define( 'SHOPWP_WEBHOOKS_MODULE_DIR', plugin_dir_path( __FILE__ ) );
}

if (!defined('SHOPWP_WEBHOOKS_BASENAME')) {
    define('SHOPWP_WEBHOOKS_BASENAME', plugin_basename(__FILE__));
}

if (!defined('SHOPWP_WEBHOOKS_MODULE_URL')) {
   define( 'SHOPWP_WEBHOOKS_MODULE_URL', plugins_url( '/', __FILE__ ) );
}

if (!defined('SHOPWP_DOWNLOAD_NAME_WEBHOOKS')) {
   define('SHOPWP_DOWNLOAD_NAME_WEBHOOKS', 'Webhooks Extension');
}

if (!defined('SHOPWP_DOWNLOAD_ID_WEBHOOKS_EXTENSION')) {
	define('SHOPWP_DOWNLOAD_ID_WEBHOOKS_EXTENSION', 238178);
}

if (!defined('SHOPWP_WEBHOOKS_NEW_PLUGIN_VERSION')) {
	define('SHOPWP_WEBHOOKS_NEW_PLUGIN_VERSION', '1.3.6');
}

final class ShopWP_Webhooks_Extension {

	const MINIMUM_PHP_VERSION = '5.6';

	public function __construct() {
		add_action( 'init', [$this, 'init']);
		add_action( 'admin_enqueue_scripts', [$this, 'enqueue_scripts']);
		add_action( 'shopwp_is_ready', [$this, 'plugin_updater']);
		add_filter(	'script_loader_tag', [$this, 'add_type_attribute'] , 10, 3);
	}

	public function i18n() {
		load_plugin_textdomain( 'shopwp-webhooks' );
	}

	public function enqueue_scripts() {

		if (!defined('SHOPWP_PLUGIN_DIR_PATH')) {
			return;
		}

		$Backend = \ShopWP\Factories\Backend_Factory::build();

		if ($Backend->should_load_js() && !$Backend->is_wizard_page() && !$Backend->is_shortcodes_page()) {
			wp_enqueue_script( 
				'shopwp_webhooks_scripts', 
				plugin_dir_url( __FILE__ ) . 'app/custom/dist/assets/index.js', 
				[
					'wp-element',
					'wp-components', 
					'shopwp-admin'
				],
				filemtime(SHOPWP_WEBHOOKS_MODULE_DIR . 'app/custom/dist/assets/index.js'), 
				true 
			);

			wp_enqueue_style('shopwp-webhooks-styles', plugin_dir_url(__FILE__) . 'app/custom/src/assets/styles.css', [], filemtime(SHOPWP_WEBHOOKS_MODULE_DIR . 'app/custom/src/assets/styles.css'));
		}
		
	}

	public function add_type_attribute($tag, $handle, $src) {
		// if not your script, do nothing and return original $tag
		if ( 'shopwp_webhooks_scripts' !== $handle ) {
			return $tag;
		}

		// change the script tag by adding type="module" and return it.
		$tag = '<script type="module" src="' . esc_url( $src ) . '"></script>';

		return $tag;
	}

	public function init() {

		if ( version_compare( PHP_VERSION, self::MINIMUM_PHP_VERSION, '<' ) ) {
			add_action( 'admin_notices', array( $this, 'admin_notice_minimum_php_version' ) );
			return;
		}

		if ( !defined('SHOPWP_BASENAME') || SHOPWP_BASENAME !== 'shopwp-pro/shopwp.php') {
			add_action( 'admin_notices', array( $this, 'admin_notice_missing_main_plugin' ) );
			return;
		}

		$this->i18n();

		require_once( 'plugin.php' );
	}

	public function plugin_updater() {

		$DB_Settings_License = ShopWP\Factories\DB\Settings_License_Factory::build();
		$DB_Settings_General = ShopWP\Factories\DB\Settings_General_Factory::build();
		$Updater = ShopWP\Factories\Updater_Factory::build();

		$license = new \stdClass();
		$license->download_id 	= 238178;
		$license->item_name 	= 'Webhooks';
		$license->license_key 	= '0961fd4ef78899af0f0b5c99d38ae070';

		$Updater->maybe_check_for_updates([
			'licenses' 			 => [$license],
			'plugin_basename'    => SHOPWP_WEBHOOKS_BASENAME,
			'current_version'    => SHOPWP_WEBHOOKS_NEW_PLUGIN_VERSION,
			'item_id'            => SHOPWP_DOWNLOAD_ID_WEBHOOKS_EXTENSION,
			'enable_beta'        => false,
			'updater_class_name' => 'ShopWP_Webhooks_EDD_SL_Plugin_Updater',
			'updater_class_path' => SHOPWP_WEBHOOKS_MODULE_DIR . 'vendor/EDD/ShopWP_Webhooks_EDD_SL_Plugin_Updater.php',
		]);

	}

	public function admin_notice_missing_main_plugin() {

		if ( isset( $_GET['activate'] ) ) {
			unset( $_GET['activate'] );
		}

		$message = sprintf(
			/* translators: 1: Plugin name 2: Recharge */
			esc_html__( '"%1$s" requires "%2$s" to be installed and activated.', 'shopwp-webhooks' ),
			'<strong>' . esc_html__( 'ShopWP Webhooks', 'shopwp-webhooks' ) . '</strong>',
			'<strong>' . esc_html__( 'ShopWP Pro', 'shopwp-webhooks' ) . '</strong>'
		);

		deactivate_plugins( plugin_basename( __FILE__ ) );

		printf( '<div class="notice notice-error is-dismissible"><p>%1$s</p></div>', $message );
	}

	public function admin_notice_minimum_php_version() {
		if ( isset( $_GET['activate'] ) ) {
			unset( $_GET['activate'] );
		}

		$message = sprintf(
			/* translators: 1: Plugin name 2: PHP 3: Required PHP version */
			esc_html__( '"%1$s" requires "%2$s" version %3$s or greater.', 'shopwp-webhooks' ),
			'<strong>' . esc_html__( 'ShopWP Yotpo Reviews Extension', 'shopwp-webhooks' ) . '</strong>',
			'<strong>' . esc_html__( 'PHP', 'shopwp-webhooks' ) . '</strong>',
			self::MINIMUM_PHP_VERSION
		);

		printf( '<div class="notice notice-warning is-dismissible"><p>%1$s</p></div>', $message );
   }

}


new ShopWP_Webhooks_Extension();

/*

Since the user will never enter a license key for this extension, it's important 
that we manually insert the record into the database so that updates will be checked for.

*/
register_activation_hook(__FILE__, function ($network_wide) {
	\ShopWP\Factories\DB\Settings_License_Factory::build()->insert_license([
		"license_key" => "0961fd4ef78899af0f0b5c99d38ae070",
		"is_local" => 0,
		"site_count" => 999,
		"checksum" => "44d3646834be31f79419c55776e4a032",
		"customer_email" => "hello@wpshop.io",
		"customer_name" => "ShopWP",
		"item_name" => "Webhooks",
		"license" => "valid",
		"license_limit" => -1,
		"payment_id" => 237832,
		"success" => 1,
		"nonce" => "",
		"activations_left" => "1",
		"is_free" => 0,
		"is_pro" => 0,
		"beta_access" => 0,
		"download_id" => 238178,
		"expires" => "1970-01-01T06:00:00.000Z"
	]);

});